<?php
require_once 'db.php';

class Note {
    private $conn;
    private $table_name = "notes";
    private $images_table = "images";
    
    // 笔记属性
    public $id;
    public $title;
    public $content;
    public $created_at;
    public $updated_at;
    
    // 构造函数
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    // 创建新笔记
    public function create() {
        try {
            // 准备SQL语句，手动设置创建和更新时间
            $query = "INSERT INTO " . $this->table_name . " (title, content, created_at, updated_at) VALUES (:title, :content, :created_at, :updated_at)";
            $stmt = $this->conn->prepare($query);
            
            // 清理和绑定数据
            $this->title = htmlspecialchars(strip_tags($this->title));
            $this->content = htmlspecialchars(strip_tags($this->content));
            
            // 设置当前时间
            $current_time = date('Y-m-d H:i:s');
            
            $stmt->bindParam(':title', $this->title);
            $stmt->bindParam(':content', $this->content);
            $stmt->bindParam(':created_at', $current_time);
            $stmt->bindParam(':updated_at', $current_time);
            
            // 执行查询
            if($stmt->execute()) {
                return $this->conn->lastInsertId();
            }
            return false;
        } catch(PDOException $e) {
            echo "创建笔记错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 保存图片
    public function saveImage($note_id, $image_path) {
        try {
            $query = "INSERT INTO " . $this->images_table . " (note_id, image_path, created_at) VALUES (:note_id, :image_path, :created_at)";
            $stmt = $this->conn->prepare($query);
            
            // 设置当前时间
            $current_time = date('Y-m-d H:i:s');
            
            $stmt->bindParam(':note_id', $note_id);
            $stmt->bindParam(':image_path', $image_path);
            $stmt->bindParam(':created_at', $current_time);
            
            return $stmt->execute();
        } catch(PDOException $e) {
            echo "保存图片错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 读取所有笔记
    public function readAll() {
        try {
            $query = "SELECT * FROM " . $this->table_name . " ORDER BY created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            
            return $stmt;
        } catch(PDOException $e) {
            echo "读取笔记错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 读取单个笔记
    public function readOne() {
        try {
            $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id LIMIT 1";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':id', $this->id);
            $stmt->execute();
            
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if($row) {
                $this->title = $row['title'];
                $this->content = $row['content'];
                $this->created_at = $row['created_at'];
                $this->updated_at = $row['updated_at'];
                return true;
            }
            return false;
        } catch(PDOException $e) {
            echo "读取单个笔记错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 获取笔记的图片
    public function getImages() {
        try {
            $query = "SELECT * FROM " . $this->images_table . " WHERE note_id = :note_id";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':note_id', $this->id);
            $stmt->execute();
            
            return $stmt;
        } catch(PDOException $e) {
            echo "获取图片错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 更新笔记
    public function update() {
        try {
            $query = "UPDATE " . $this->table_name . " 
                      SET title = :title, content = :content, updated_at = :updated_at
                      WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            
            // 清理和绑定数据
            $this->title = htmlspecialchars(strip_tags($this->title));
            $this->content = htmlspecialchars(strip_tags($this->content));
            
            // 设置当前时间作为更新时间
            $current_time = date('Y-m-d H:i:s');
            
            $stmt->bindParam(':title', $this->title);
            $stmt->bindParam(':content', $this->content);
            $stmt->bindParam(':updated_at', $current_time);
            $stmt->bindParam(':id', $this->id);
            
            return $stmt->execute();
        } catch(PDOException $e) {
            echo "更新笔记错误: " . $e->getMessage();
            return false;
        }
    }
    
    // 删除笔记
    public function delete() {
        try {
            // 删除笔记前先删除相关的图片文件
            $images = $this->getImages();
            while($row = $images->fetch(PDO::FETCH_ASSOC)) {
                if(file_exists($row['image_path'])) {
                    unlink($row['image_path']);
                }
            }
            
            // 删除笔记（外键约束会自动删除相关的图片记录）
            $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':id', $this->id);
            
            return $stmt->execute();
        } catch(PDOException $e) {
            echo "删除笔记错误: " . $e->getMessage();
            return false;
        }
    }
}
?> 