<?php
// 包含数据库和笔记类文件
require_once 'note.php';
require_once 'auth.php'; // 添加身份验证

// 检查ID参数
if(!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: index.php");
    exit;
}

// 创建笔记对象
$note = new Note();
$note->id = $_GET['id'];

// 获取笔记详情
if(!$note->readOne()) {
    header("Location: index.php?error=笔记不存在");
    exit;
}

// 获取笔记的图片
$images = $note->getImages();

// 处理表单提交
if($_SERVER["REQUEST_METHOD"] == "POST") {
    // 设置笔记属性
    $note->title = $_POST['title'];
    $note->content = $_POST['content'];
    
    // 更新笔记
    if($note->update()) {
        // 处理图片上传
        $upload_dir = "uploads/";
        
        // 创建上传目录（如果不存在）
        if(!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        // 处理多个图片上传
        if(isset($_FILES['images']) && !empty($_FILES['images']['name'][0])) {
            $total_files = count($_FILES['images']['name']);
            
            for($i = 0; $i < $total_files; $i++) {
                // 检查文件上传是否成功
                if($_FILES['images']['error'][$i] == 0) {
                    $temp_name = $_FILES['images']['tmp_name'][$i];
                    $file_name = $_FILES['images']['name'][$i];
                    
                    // 生成唯一文件名
                    $new_file_name = $upload_dir . uniqid() . '_' . basename($file_name);
                    
                    // 检查文件类型
                    $allowed_types = array('image/jpeg', 'image/png', 'image/gif');
                    $file_type = $_FILES['images']['type'][$i];
                    
                    if(in_array($file_type, $allowed_types)) {
                        // 移动上传的文件
                        if(move_uploaded_file($temp_name, $new_file_name)) {
                            // 保存图片路径到数据库
                            $note->saveImage($note->id, $new_file_name);
                        }
                    }
                }
            }
        }
        
        // 删除选定的图片
        if(isset($_POST['delete_image']) && !empty($_POST['delete_image'])) {
            foreach($_POST['delete_image'] as $image_id => $value) {
                // 通过一个简单的查询获取图片路径
                try {
                    $db = new Database();
                    $conn = $db->getConnection();
                    
                    $query = "SELECT image_path FROM images WHERE id = :id";
                    $stmt = $conn->prepare($query);
                    $stmt->bindParam(':id', $image_id);
                    $stmt->execute();
                    
                    if($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        // 删除物理文件
                        if(file_exists($row['image_path'])) {
                            unlink($row['image_path']);
                        }
                        
                        // 从数据库中删除记录
                        $query = "DELETE FROM images WHERE id = :id";
                        $stmt = $conn->prepare($query);
                        $stmt->bindParam(':id', $image_id);
                        $stmt->execute();
                    }
                } catch(PDOException $e) {
                    // 处理错误（此处简单打印）
                    error_log("删除图片错误: " . $e->getMessage());
                }
            }
        }
        
        // 重定向到查看页面
        header("Location: view.php?id=" . $note->id . "&success=笔记更新成功");
        exit;
    } else {
        $error_message = "更新笔记失败，请稍后重试";
    }
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>编辑笔记 - 在线记事本</title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f5f5f5;
            color: #333;
            line-height: 1.6;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            padding: 20px;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        h1 {
            font-size: 24px;
            color: #333;
        }
        .btn {
            display: inline-block;
            padding: 8px 16px;
            background-color: #4CAF50;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            font-size: 14px;
            transition: background-color 0.3s;
            border: none;
            cursor: pointer;
        }
        .btn:hover {
            background-color: #45a049;
        }
        .btn-outline {
            background-color: transparent;
            border: 1px solid #4CAF50;
            color: #4CAF50;
        }
        .btn-outline:hover {
            background-color: #4CAF50;
            color: white;
        }
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
        }
        .alert-danger {
            background-color: #f2dede;
            color: #a94442;
            border: 1px solid #ebccd1;
        }
        .card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            padding: 20px;
            margin-bottom: 20px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: bold;
        }
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 16px;
            font-family: inherit;
        }
        .form-text {
            font-size: 14px;
            color: #777;
            margin-top: 5px;
        }
        textarea.form-control {
            min-height: 150px;
            resize: vertical;
        }
        .preview-image {
            max-width: 150px;
            max-height: 150px;
            margin-right: 10px;
            margin-bottom: 10px;
            border-radius: 5px;
            object-fit: cover;
        }
        .image-preview {
            display: flex;
            flex-wrap: wrap;
            margin-top: 10px;
        }
        .existing-images {
            display: flex;
            flex-wrap: wrap;
            margin-top: 10px;
            margin-bottom: 15px;
        }
        .existing-image {
            position: relative;
            display: inline-block;
            margin-right: 15px;
            margin-bottom: 15px;
        }
        .existing-image img {
            width: 120px;
            height: 120px;
            object-fit: cover;
            border-radius: 4px;
        }
        .image-delete {
            position: absolute;
            top: 5px;
            right: 5px;
            background: rgba(255,255,255,0.8);
            border-radius: 50%;
            width: 22px;
            height: 22px;
            text-align: center;
            line-height: 22px;
            cursor: pointer;
            color: #dc3545;
            font-weight: bold;
        }
        .marked-for-delete {
            opacity: 0.5;
        }
        .button-group {
            display: flex;
            gap: 10px;
        }
        .close-btn {
            float: right;
            font-size: 20px;
            font-weight: bold;
            cursor: pointer;
        }
        @media (max-width: 768px) {
            .header {
                flex-direction: column;
                align-items: start;
                gap: 10px;
            }
            .button-group {
                flex-direction: column;
                width: 100%;
            }
            .button-group a {
                margin-bottom: 8px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>编辑笔记</h1>
            <div class="button-group">
                <a href="view.php?id=<?php echo $note->id; ?>" class="btn btn-outline">取消</a>
                <a href="index.php" class="btn btn-outline">返回列表</a>
            </div>
        </div>
        
        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger">
                <?php echo $error_message; ?>
                <span class="close-btn" onclick="this.parentElement.style.display='none'">&times;</span>
            </div>
        <?php endif; ?>
        
        <div class="card">
            <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . '?id=' . $note->id; ?>" method="post" enctype="multipart/form-data">
                <div class="form-group">
                    <label for="title" class="form-label">标题</label>
                    <input type="text" class="form-control" id="title" name="title" value="<?php echo htmlspecialchars($note->title); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="content" class="form-label">内容</label>
                    <textarea class="form-control" id="content" name="content" rows="6" required><?php echo htmlspecialchars($note->content); ?></textarea>
                </div>
                
                <?php if($images && $images->rowCount() > 0): ?>
                    <div class="form-group">
                        <label class="form-label">现有图片</label>
                        <div class="existing-images">
                            <?php while($img = $images->fetch(PDO::FETCH_ASSOC)): ?>
                                <div class="existing-image">
                                    <img src="<?php echo htmlspecialchars($img['image_path']); ?>" alt="附件图片">
                                    <div class="image-delete" title="删除此图片" onclick="toggleImageDelete(this, <?php echo $img['id']; ?>)">×</div>
                                    <input type="hidden" name="delete_image[<?php echo $img['id']; ?>]" value="0">
                                </div>
                            <?php endwhile; ?>
                        </div>
                        <div class="form-text">点击图片右上角的 × 可标记要删除的图片</div>
                    </div>
                <?php endif; ?>
                
                <div class="form-group">
                    <label for="images" class="form-label">上传新图片（可选，支持多张）</label>
                    <input type="file" class="form-control" id="images" name="images[]" multiple accept="image/*">
                    <div class="form-text">支持JPG、PNG和GIF格式，可同时上传多张图片</div>
                    <div id="imagePreview" class="image-preview"></div>
                </div>
                
                <button type="submit" class="btn">更新笔记</button>
            </form>
        </div>
    </div>
    
    <script>
        // 图片预览功能
        document.getElementById('images').addEventListener('change', function(event) {
            const preview = document.getElementById('imagePreview');
            preview.innerHTML = '';
            
            for (let i = 0; i < event.target.files.length; i++) {
                const file = event.target.files[i];
                if (!file.type.startsWith('image/')) continue;
                
                const img = document.createElement('img');
                img.classList.add('preview-image');
                img.file = file;
                preview.appendChild(img);
                
                const reader = new FileReader();
                reader.onload = (function(aImg) { 
                    return function(e) { 
                        aImg.src = e.target.result; 
                    }; 
                })(img);
                reader.readAsDataURL(file);
            }
        });
        
        // 标记/取消标记要删除的图片
        function toggleImageDelete(element, imageId) {
            const parent = element.parentElement;
            const input = parent.querySelector('input[type="hidden"]');
            
            if (parent.classList.contains('marked-for-delete')) {
                parent.classList.remove('marked-for-delete');
                parent.style.opacity = '1';
                input.value = '0';
            } else {
                parent.classList.add('marked-for-delete');
                parent.style.opacity = '0.5';
                input.value = '1';
            }
        }
        
        // 关闭提示框功能
        document.addEventListener('DOMContentLoaded', function() {
            const closeButtons = document.querySelectorAll('.close-btn');
            closeButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    this.parentElement.style.display = 'none';
                });
            });
        });
    </script>
</body>
</html> 